#include "unicode_symbols.h"
#include "keyboard.h"
#include "translate.h"

struct unicode_close_match {
	t_unicode symbol;  /* A unicode symbol for which we have an approximation */
	t_unicode approximation; /*  A symbol that closely resembles symbol */
};

/* Note: order is important in this list,
 * For a given symbol the approximations that is closes should
 * come first.
 */
static const struct unicode_close_match approximations[] =
{
	/* Approximation for unmapped characters */
	{ U_REPLACEMENT_CHARACTER,	U_QUESTION_MARK },

	/* Approximations on dosemus special keysyms
	 * in the private unicode area.
	 */
	{ KEY_PAD_0,		U_DIGIT_ZERO },
	{ KEY_PAD_1,		U_DIGIT_ONE },
	{ KEY_PAD_2,		U_DIGIT_TWO },
	{ KEY_PAD_3,		U_DIGIT_THREE },
	{ KEY_PAD_4,		U_DIGIT_FOUR },
	{ KEY_PAD_5,		U_DIGIT_FIVE },
	{ KEY_PAD_6,		U_DIGIT_SIX },
	{ KEY_PAD_7,		U_DIGIT_SEVEN },
	{ KEY_PAD_8,		U_DIGIT_EIGHT },
	{ KEY_PAD_9,		U_DIGIT_NINE },
	{ KEY_PAD_DECIMAL,	U_PERIOD },
#ifdef HAVE_UNICODE_KEYB
	{ KEY_PAD_SEPARATOR,	U_COMMA },
#endif
	{ KEY_PAD_SLASH,	U_SLASH },
	{ KEY_PAD_AST,		U_ASTERISK },
	{ KEY_PAD_MINUS,	U_HYPHEN_MINUS },
	{ KEY_PAD_PLUS,		U_PLUS_SIGN },
	{ KEY_PAD_ENTER,	U_CARRIAGE_RETURN },

	/* Be careful with these KEY_XXXX -> U_NULL mappings.
	 * These are crucial for the keyboard code to work.
	 * These mappings prevent unmapped function keys from
	 * having the same mapping as unmapped characters,
	 * which are o.k. to display as a strange character.
	 *
	 */
	{ KEY_PAD_HOME,		U_NULL },
	{ KEY_PAD_UP,		U_NULL },
	{ KEY_PAD_PGUP,		U_NULL },

	{ KEY_PAD_LEFT,		U_NULL },
	{ KEY_PAD_CENTER,	U_NULL },
	{ KEY_PAD_RIGHT,	U_NULL },
	{ KEY_PAD_END,		U_NULL },
	{ KEY_PAD_DOWN,		U_NULL },
	{ KEY_PAD_PGDN,		U_NULL },
	{ KEY_PAD_INS,		U_NULL },
	{ KEY_PAD_DEL,		U_NULL },


	{ KEY_ESC,		U_NULL },
	{ KEY_F1,		U_NULL },
	{ KEY_F2,		U_NULL },
	{ KEY_F3,		U_NULL },
	{ KEY_F4,		U_NULL },
	{ KEY_F5,		U_NULL },
	{ KEY_F6,		U_NULL },
	{ KEY_F7,		U_NULL },
	{ KEY_F8,		U_NULL },
	{ KEY_F9,		U_NULL },
	{ KEY_F10,		U_NULL },
	{ KEY_F11,		U_NULL },
	{ KEY_F12,		U_NULL },


	{ KEY_INS,		U_NULL },
	{ KEY_DEL,		U_NULL },
	{ KEY_HOME,		U_NULL },
	{ KEY_END,		U_NULL },
	{ KEY_PGUP,		U_NULL },
	{ KEY_PGDN,		U_NULL },
	{ KEY_UP,		U_NULL },
	{ KEY_DOWN,		U_NULL },
	{ KEY_LEFT,		U_NULL },
	{ KEY_RIGHT,		U_NULL },

	{ KEY_L_ALT,		U_NULL },
	{ KEY_R_ALT,		U_NULL },
	{ KEY_L_CTRL,		U_NULL },
	{ KEY_R_CTRL,		U_NULL },
	{ KEY_L_SHIFT,		U_NULL },
	{ KEY_R_SHIFT,		U_NULL },
	{ KEY_NUM,		U_NULL },
	{ KEY_SCROLL,		U_NULL },
	{ KEY_CAPS,		U_NULL },

	{ KEY_PRTSCR,		U_NULL },
	{ KEY_PAUSE,		U_NULL },
	{ KEY_SYSRQ,		U_NULL },
	{ KEY_BREAK,		U_NULL },

	{ KEY_PAD_SEPARATOR,	U_NULL },

	{ KEY_ALT_A,		U_NULL },
	{ KEY_ALT_B,		U_NULL },
	{ KEY_ALT_C,		U_NULL },
	{ KEY_ALT_D,		U_NULL },
	{ KEY_ALT_E,		U_NULL },
	{ KEY_ALT_F,		U_NULL },
	{ KEY_ALT_G,		U_NULL },
	{ KEY_ALT_H,		U_NULL },
	{ KEY_ALT_I,		U_NULL },
	{ KEY_ALT_J,		U_NULL },
	{ KEY_ALT_K,		U_NULL },
	{ KEY_ALT_L,		U_NULL },
	{ KEY_ALT_M,		U_NULL },
	{ KEY_ALT_N,		U_NULL },
	{ KEY_ALT_O,		U_NULL },
	{ KEY_ALT_P,		U_NULL },
	{ KEY_ALT_Q,		U_NULL },
	{ KEY_ALT_R,		U_NULL },
	{ KEY_ALT_S,		U_NULL },
	{ KEY_ALT_T,		U_NULL },
	{ KEY_ALT_U,		U_NULL },
	{ KEY_ALT_V,		U_NULL },
	{ KEY_ALT_W,		U_NULL },
	{ KEY_ALT_X,		U_NULL },
	{ KEY_ALT_Y,		U_NULL },
	{ KEY_ALT_Z,		U_NULL },

	{ KEY_LEFT_TAB,		U_NULL },

	/* General approximations */
	{ U_HOUSE,				U_CIRCUMFLEX_ACCENT },

	{ U_WHITE_SMILING_FACE,			U_LATIN_SMALL_LETTER_O_WITH_DIAERESIS },
	{ U_BLACK_SMILING_FACE, 		U_LATIN_CAPITAL_LETTER_O_WITH_DIAERESIS },
	{ U_BLACK_HEART_SUIT,			U_LATIN_SMALL_LETTER_H },
	{ U_BLACK_DIAMOND_SUIT,			U_LATIN_SMALL_LETTER_D },
	{ U_BLACK_CLUB_SUIT,			U_LATIN_SMALL_LETTER_C },
	{ U_BLACK_SPADE_SUIT,			U_LATIN_SMALL_LETTER_S },

 	{ U_BULLET,				U_BULLET_OPERATOR },
 	{ U_BULLET,				U_DEGREE_SIGN },
 	{ U_BULLET,				U_MIDDLE_DOT },
	{ U_BULLET,				U_PLUS_SIGN },
	{ U_INVERSE_BULLET,			U_DARK_SHADE },
	{ U_INVERSE_BULLET,			U_NUMBER_SIGN },
	{ U_WHITE_CIRCLE, 			U_LATIN_SMALL_LETTER_O },
	{ U_INVERSE_WHITE_CIRCLE,		U_MEDIUM_SHADE },
	{ U_INVERSE_WHITE_CIRCLE,		U_DIGIT_ZERO },
	{ U_MALE_SIGN, 				U_LATIN_SMALL_LETTER_O_WITH_ACUTE },
	{ U_FEMALE_SIGN, 			U_YEN_SIGN },
	{ U_FEMALE_SIGN, 			U_PLUS_SIGN },
	{ U_EIGHTH_NOTE, 			U_SQUARE_ROOT },
	{ U_EIGHTH_NOTE, 			U_SEMICOLON },
	{ U_BEAMED_EIGHTH_NOTES,		U_INTERSECTION },
	{ U_BEAMED_EIGHTH_NOTES,		U_LATIN_CAPITAL_LETTER_M },
	{ U_WHITE_SUN_WITH_RAYS,		U_ASTERISK },

	{ U_BLACK_RIGHT_POINTING_POINTER,	U_GREATER_THAN_SIGN},
	{ U_BLACK_LEFT_POINTING_POINTER,	U_LESS_THAN_SIGN},
	{ U_BLACK_RIGHT_POINTING_SMALL_TRIANGLE,	U_GREATER_THAN_SIGN},
	{ U_BLACK_LEFT_POINTING_SMALL_TRIANGLE,	U_LESS_THAN_SIGN},
	{ U_UP_DOWN_ARROW, 			U_INVERTED_EXCLAMATION_MARK},
	{ U_UP_DOWN_ARROW,			U_VERTICAL_LINE},
	{ U_DOUBLE_EXCLAMATION_MARK,		U_BOX_DRAWINGS_DOUBLE_VERTICAL},
	{ U_DOUBLE_EXCLAMATION_MARK,		U_LATIN_CAPITAL_LETTER_H},
	{ U_PILCROW_SIGN, 			U_GREEK_SMALL_LETTER_PI},
	{ U_PILCROW_SIGN, 			U_LATIN_CAPITAL_LETTER_P},
	{ U_SECTION_SIGN,			U_POUND_SIGN},
	{ U_SECTION_SIGN,			U_LATIN_SMALL_LETTER_F_WITH_HOOK},
	{ U_BLACK_RECTANGLE, 			U_BLACK_SQUARE},
	{ U_BLACK_RECTANGLE, 			U_PLUS_SIGN},
	{ U_UP_DOWN_ARROW_WITH_BASE,		U_INVERTED_QUESTION_MARK},
	{ U_UP_DOWN_ARROW_WITH_BASE,		U_VERTICAL_LINE},
	{ U_UPWARDS_ARROW, 			U_TOP_HALF_INTEGRAL},
	{ U_UPWARDS_ARROW, 			U_CIRCUMFLEX_ACCENT},
	{ U_DOWNWARDS_ARROW,			U_BOTTOM_HALF_INTEGRAL},
	{ U_DOWNWARDS_ARROW,			U_LATIN_SMALL_LETTER_V},
	{ U_RIGHTWARDS_ARROW,			U_RIGHT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK},
	{ U_RIGHTWARDS_ARROW,			U_GREATER_THAN_SIGN},
	{ U_LEFTWARDS_ARROW,			U_LEFT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK},
	{ U_LEFTWARDS_ARROW,			U_LESS_THAN_SIGN},
	{ U_RIGHT_ANGLE,			U_BOX_DRAWINGS_LIGHT_UP_AND_RIGHT},
	{ U_RIGHT_ANGLE,			U_HYPHEN_MINUS},
	{ U_LEFT_RIGHT_ARROW,			U_BOX_DRAWINGS_DOUBLE_HORIZONTAL},
	{ U_LEFT_RIGHT_ARROW,			U_HYPHEN_MINUS},
	{ U_BLACK_UP_POINTING_TRIANGLE,		U_CIRCUMFLEX_ACCENT},
	{ U_BLACK_DOWN_POINTING_TRIANGLE,	U_LATIN_SMALL_LETTER_V},

	{ U_PESETA_SIGN,			U_LATIN_CAPITAL_LETTER_P},
	{ U_LATIN_SMALL_LETTER_F_WITH_HOOK,	U_LATIN_SMALL_LETTER_F},

	{ U_REVERSED_NOT_SIGN,			U_HYPHEN_MINUS },

	{ U_LIGHT_SHADE, 			U_COLON },
	{ U_MEDIUM_SHADE,			U_PERCENT_SIGN},
	{ U_DARK_SHADE,				U_AMPERSAND},
	{ U_BOX_DRAWINGS_LIGHT_VERTICAL,			U_VERTICAL_LINE},
	{ U_BOX_DRAWINGS_LIGHT_VERTICAL_AND_LEFT,		U_LEFT_CURLY_BRACKET},
	{ U_BOX_DRAWINGS_VERTICAL_SINGLE_AND_LEFT_DOUBLE,	U_LEFT_CURLY_BRACKET},
	{ U_BOX_DRAWINGS_VERTICAL_DOUBLE_AND_LEFT_SINGLE,	U_LEFT_CURLY_BRACKET},
	{ U_BOX_DRAWINGS_DOWN_DOUBLE_AND_LEFT_SINGLE,		U_PERIOD},
	{ U_BOX_DRAWINGS_DOWN_SINGLE_AND_LEFT_DOUBLE,		U_PERIOD},
	{ U_BOX_DRAWINGS_DOUBLE_VERTICAL_AND_LEFT,		U_LEFT_CURLY_BRACKET},
	{ U_BOX_DRAWINGS_DOUBLE_VERTICAL, 			U_LATIN_CAPITAL_LETTER_I},
	{ U_BOX_DRAWINGS_DOUBLE_DOWN_AND_LEFT,			U_PERIOD},
	{ U_BOX_DRAWINGS_DOUBLE_UP_AND_LEFT,			U_APOSTROPHE},
	{ U_BOX_DRAWINGS_UP_DOUBLE_AND_LEFT_SINGLE,		U_APOSTROPHE},
	{ U_BOX_DRAWINGS_UP_SINGLE_AND_LEFT_DOUBLE,		U_APOSTROPHE},
	{ U_BOX_DRAWINGS_LIGHT_DOWN_AND_LEFT,			U_NOT_SIGN},
	{ U_BOX_DRAWINGS_LIGHT_DOWN_AND_LEFT,			U_PERIOD},

	{ U_BOX_DRAWINGS_LIGHT_UP_AND_RIGHT,			U_GRAVE_ACCENT},
	{ U_BOX_DRAWINGS_LIGHT_UP_AND_HORIZONTAL,		U_GRAVE_ACCENT},
	{ U_BOX_DRAWINGS_LIGHT_DOWN_AND_HORIZONTAL,		U_PLUS_SIGN},
	{ U_BOX_DRAWINGS_LIGHT_VERTICAL_AND_RIGHT,		U_RIGHT_CURLY_BRACKET},
	{ U_BOX_DRAWINGS_LIGHT_HORIZONTAL,			U_HYPHEN_MINUS},
	{ U_BOX_DRAWINGS_LIGHT_VERTICAL_AND_HORIZONTAL,		U_PLUS_SIGN},
	{ U_BOX_DRAWINGS_VERTICAL_SINGLE_AND_RIGHT_DOUBLE,	U_RIGHT_CURLY_BRACKET},
	{ U_BOX_DRAWINGS_VERTICAL_DOUBLE_AND_RIGHT_SINGLE,	U_RIGHT_CURLY_BRACKET},
	{ U_BOX_DRAWINGS_DOUBLE_UP_AND_RIGHT,			U_GRAVE_ACCENT},
	{ U_BOX_DRAWINGS_DOUBLE_DOWN_AND_RIGHT,			U_PERIOD},
	{ U_BOX_DRAWINGS_DOUBLE_UP_AND_HORIZONTAL,		U_ASTERISK},
	{ U_BOX_DRAWINGS_DOUBLE_DOWN_AND_HORIZONTAL,		U_ASTERISK},
	{ U_BOX_DRAWINGS_DOUBLE_VERTICAL_AND_RIGHT,		U_RIGHT_CURLY_BRACKET},
	{ U_BOX_DRAWINGS_DOUBLE_HORIZONTAL,			U_EQUALS_SIGN},
	{ U_BOX_DRAWINGS_DOUBLE_VERTICAL_AND_HORIZONTAL,	U_ASTERISK},
	{ U_BOX_DRAWINGS_UP_SINGLE_AND_HORIZONTAL_DOUBLE,	U_ASTERISK},

	{ U_BOX_DRAWINGS_UP_DOUBLE_AND_HORIZONTAL_SINGLE,	U_PLUS_SIGN},
	{ U_BOX_DRAWINGS_DOWN_SINGLE_AND_HORIZONTAL_DOUBLE,	U_ASTERISK},
	{ U_BOX_DRAWINGS_DOWN_DOUBLE_AND_HORIZONTAL_SINGLE,	U_PLUS_SIGN},
	{ U_BOX_DRAWINGS_UP_DOUBLE_AND_RIGHT_SINGLE,		U_GRAVE_ACCENT},
	{ U_BOX_DRAWINGS_UP_SINGLE_AND_RIGHT_DOUBLE,		U_GRAVE_ACCENT},
	{ U_BOX_DRAWINGS_DOWN_SINGLE_AND_RIGHT_DOUBLE,		U_PERIOD},
	{ U_BOX_DRAWINGS_DOWN_DOUBLE_AND_RIGHT_SINGLE,		U_PERIOD},
	{ U_BOX_DRAWINGS_VERTICAL_DOUBLE_AND_HORIZONTAL_SINGLE,	U_ASTERISK},
	{ U_BOX_DRAWINGS_VERTICAL_SINGLE_AND_HORIZONTAL_DOUBLE,	U_ASTERISK},
	{ U_BOX_DRAWINGS_LIGHT_UP_AND_LEFT,			U_APOSTROPHE},
	{ U_BOX_DRAWINGS_LIGHT_DOWN_AND_RIGHT,			U_PERIOD},
	{ U_FULL_BLOCK,				U_NUMBER_SIGN},
	{ U_LOWER_HALF_BLOCK,			U_LOW_LINE},
	{ U_LEFT_HALF_BLOCK,			U_RIGHT_SQUARE_BRACKET},
	{ U_RIGHT_HALF_BLOCK,			U_LEFT_SQUARE_BRACKET},
	{ U_UPPER_HALF_BLOCK,			U_TILDE},

	{ U_GREEK_SMALL_LETTER_ALPHA,		U_LATIN_SMALL_LETTER_A},

	{ U_GREEK_CAPITAL_LETTER_GAMMA,		U_NOT_SIGN},
	{ U_GREEK_SMALL_LETTER_PI,		U_PILCROW_SIGN},
	{ U_GREEK_CAPITAL_LETTER_SIGMA,		U_LEFT_CURLY_BRACKET},
	{ U_GREEK_SMALL_LETTER_SIGMA,		U_LATIN_SMALL_LETTER_O_WITH_ACUTE},

	{ U_GREEK_SMALL_LETTER_TAU,		U_LATIN_SMALL_LETTER_T},
	{ U_GREEK_CAPITAL_LETTER_PHI,		U_LATIN_CAPITAL_LETTER_O_WITH_STROKE},
	{ U_GREEK_CAPITAL_LETTER_THETA,		U_LATIN_SMALL_LETTER_O_WITH_CIRCUMFLEX},
	{ U_GREEK_CAPITAL_LETTER_OMEGA,		U_LATIN_CAPITAL_LETTER_O_WITH_DIAERESIS},
	{ U_GREEK_SMALL_LETTER_DELTA,		U_LATIN_SMALL_LETTER_O_WITH_ACUTE},
	{ U_INFINITY,				U_LATIN_SMALL_LETTER_O},
	{ U_GREEK_SMALL_LETTER_PHI,		U_LATIN_SMALL_LETTER_O_WITH_STROKE},
	{ U_GREEK_SMALL_LETTER_EPSILON, 	U_LATIN_CAPITAL_LETTER_E},
	{ U_INTERSECTION,			U_LATIN_SMALL_LETTER_N},

	{ U_IDENTICAL_TO,			U_EQUALS_SIGN},

	{ U_GREATER_THAN_OR_EQUAL_TO,		U_GREATER_THAN_SIGN},
	{ U_LESS_THAN_OR_EQUAL_TO,		U_LESS_THAN_SIGN},
	{ U_TOP_HALF_INTEGRAL,			U_LEFT_PARENTHESIS},
	{ U_BOTTOM_HALF_INTEGRAL,		U_RIGHT_PARENTHESIS},

	{ U_ALMOST_EQUAL_TO,			U_EQUALS_SIGN},

	{ U_BULLET_OPERATOR,			U_MIDDLE_DOT},

	{ U_SQUARE_ROOT, 			U_PERCENT_SIGN},
	{ U_SUPERSCRIPT_LATIN_SMALL_LETTER_N,	U_SUPERSCRIPT_THREE},
	{ U_SUPERSCRIPT_LATIN_SMALL_LETTER_N,	U_GRAVE_ACCENT},

	{ U_BLACK_SQUARE,			U_EQUALS_SIGN },

	{ U_LATIN_SMALL_LETTER_DOTLESS_I,	U_PERIOD},
	{ U_DOUBLE_LOW_LINE,			U_GREATER_THAN_SIGN },

	{ U_NOT_SIGN,					U_HYPHEN_MINUS},
	{ U_LEFT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK,	U_LESS_THAN_SIGN },
	{ U_RIGHT_POINTING_DOUBLE_ANGLE_QUOTATION_MARK, U_GREATER_THAN_SIGN },

	/* reverse map */
	{ U_BROKEN_BAR,				U_VERTICAL_LINE },
	{ U_DIAERESIS,				U_QUOTATION_MARK },
	{ U_SOFT_HYPHEN,			U_HYPHEN_MINUS },
	{ U_SUPERSCRIPT_THREE,			U_SUPERSCRIPT_LATIN_SMALL_LETTER_N },
	{ U_ACUTE_ACCENT,			U_APOSTROPHE },
	{ U_MIDDLE_DOT,				U_BULLET_OPERATOR },
	{ U_CEDILLA,				U_COMMA },
	{ U_LATIN_CAPITAL_LETTER_O_WITH_STROKE,	U_GREEK_SMALL_LETTER_PHI },
	{ U_LATIN_SMALL_LETTER_O_WITH_STROKE,	U_GREEK_SMALL_LETTER_PHI },
/*
	{ U_CURRENCY_SIGN,			x },
	{ U_COPYRIGHT_SIGN,			x },
	{ U_REGISTERED_SIGN,			x },
	{ U_MACRON,				x },
	{ U_SUPERSCRIPT_ONE,			x },
	{ U_VULGAR_FRACTION_THREE_QUARTERS,	x },
*/
	{ U_LATIN_CAPITAL_LETTER_A_WITH_GRAVE,	U_LATIN_CAPITAL_LETTER_A },
	{ U_LATIN_CAPITAL_LETTER_A_WITH_ACUTE,  U_LATIN_CAPITAL_LETTER_A },
	{ U_LATIN_CAPITAL_LETTER_A_WITH_CIRCUMFLEX,	U_LATIN_CAPITAL_LETTER_A },
	{ U_LATIN_CAPITAL_LETTER_A_WITH_TILDE,	U_LATIN_CAPITAL_LETTER_A },
	{ U_LATIN_CAPITAL_LETTER_E_WITH_GRAVE,  U_LATIN_CAPITAL_LETTER_E },
	{ U_LATIN_CAPITAL_LETTER_E_WITH_CIRCUMFLEX,	U_LATIN_CAPITAL_LETTER_E },
	{ U_LATIN_CAPITAL_LETTER_E_WITH_DIAERESIS,	U_LATIN_CAPITAL_LETTER_E },
	{ U_LATIN_CAPITAL_LETTER_I_WITH_GRAVE,	U_LATIN_CAPITAL_LETTER_I },
	{ U_LATIN_CAPITAL_LETTER_I_WITH_ACUTE,	U_LATIN_CAPITAL_LETTER_I },
	{ U_LATIN_CAPITAL_LETTER_I_WITH_CIRCUMFLEX,	U_LATIN_CAPITAL_LETTER_I },
	{ U_LATIN_CAPITAL_LETTER_I_WITH_DIAERESIS,	U_LATIN_CAPITAL_LETTER_I },
	{ U_LATIN_CAPITAL_LETTER_ETH,		U_LATIN_SMALL_LETTER_D },
	{ U_LATIN_CAPITAL_LETTER_O_WITH_GRAVE,	U_LATIN_CAPITAL_LETTER_O },
	{ U_LATIN_CAPITAL_LETTER_O_WITH_ACUTE,	U_LATIN_CAPITAL_LETTER_O },
	{ U_LATIN_CAPITAL_LETTER_O_WITH_CIRCUMFLEX,	U_LATIN_CAPITAL_LETTER_O },
	{ U_LATIN_CAPITAL_LETTER_O_WITH_TILDE,	U_LATIN_CAPITAL_LETTER_O },
	{ U_MULTIPLICATION_SIGN,		U_LATIN_SMALL_LETTER_X },
	{ U_LATIN_CAPITAL_LETTER_U_WITH_GRAVE,	U_LATIN_CAPITAL_LETTER_U },
	{ U_LATIN_CAPITAL_LETTER_U_WITH_ACUTE,	U_LATIN_CAPITAL_LETTER_U },
	{ U_LATIN_CAPITAL_LETTER_THORN,		U_LATIN_CAPITAL_LETTER_P },
	{ U_LATIN_SMALL_LETTER_SHARP_S,		U_LATIN_SMALL_LETTER_S },
	{ U_LATIN_SMALL_LETTER_A_WITH_TILDE,	U_LATIN_SMALL_LETTER_A },
	{ U_LATIN_SMALL_LETTER_ETH,		U_LATIN_SMALL_LETTER_D },
	{ U_LATIN_SMALL_LETTER_O_WITH_TILDE,	U_LATIN_SMALL_LETTER_O },
	{ U_LATIN_SMALL_LETTER_Y_WITH_ACUTE,	U_LATIN_SMALL_LETTER_Y },
	{ U_LATIN_SMALL_LETTER_THORN,		U_LATIN_SMALL_LETTER_P }
};
static const int approximation_count = sizeof(approximations)/sizeof(approximations[0]);

void traverse_approximations(t_unicode symbol, void *p, approximations_callback_t callback)
{
	int i;
	for(i = 0; i < approximation_count; i++) {
		if (approximations[i].symbol == symbol) {
			callback(p, approximations[i].approximation);
		}
	}
	return;
}

void traverse_approximation_list(void *p, approximation_list_callback_t callback)
{
	int i;
	for(i = 0; i < approximation_count; i++) {
		callback(p,
			approximations[i].symbol,
			approximations[i].approximation);
	}
}
